/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=24MHz, SysCLK=24MHz
*	 Description:
*			TWSI0 (SCL/P40,	SDA/P41) Slave interrupt
*			TWSI1 (SCL/P10,	SDA/P11) Slave interrupt
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		24000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)


#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	IO_TWI0_SCL		P40
#define	IO_TWI0_SDA		P41

#define	IO_TWI1_SCL		P10
#define	IO_TWI1_SDA		P11

#define	SLAVE_ADDRESS	0xA0
#define	TEST_BUF_SIZE	16

#define	TWI_OV_TIME_MAX			20

u8	TWI0OvTime;
u8	TWI1OvTime;
bit	bTWI0Error;
bit	bTWI1Error;
bit	bTWI0ADFlag;
bit	bTWI1ADFlag;

bit	bTWI0RXFlag;
bit	bTWI1RXFlag;

WordTypeDef	TWI0RegAddr;
WordTypeDef	TWI1RegAddr;
xdata u8 TWI0Buf[TEST_BUF_SIZE];
xdata u8 TWI1Buf[TEST_BUF_SIZE];
u8 TWI0Inx;
u8 TWI1Inx;
bit	bES0;
u8 LedTime;

/***********************************************************************************
*Function:		void INT_T0(void)
*Description:	T0 Interrupt handler
*Input:
*Output:
*************************************************************************************/
void INT_T0(void) interrupt	INT_VECTOR_T0
{
	TH0=TIMER_12T_1ms_TH;
	TL0=TIMER_12T_1ms_TL;
	if(TWI0OvTime!=0)
	{
		TWI0OvTime--;
		if(TWI0OvTime==0) bTWI0Error=TRUE;
	}
	if(TWI1OvTime!=0)
	{
		TWI1OvTime--;
		if(TWI1OvTime==0) bTWI1Error=TRUE;
	}
	if(LedTime!=0) LedTime--;
}


/***********************************************************************************
*Function:		void INT_TWI0(void)
*Description:	TWI0 Interrupt handler

*Input:
*Output:
*************************************************************************************/
void INT_TWI0()	interrupt INT_VECTOR_TWI0
{
	_push_(SFRPI);
	SFR_SetPage(0);
	if(SISTA==0x80)
	{ // 0x80: DATA	has	been received. ACK has been	returned.
		TWI0OvTime = TWI_OV_TIME_MAX;			// restore TWI0	overtime
		if(bTWI0ADFlag==0)
		{//	Reg	Address
			if(TWI0Inx==0)
			{
				TWI0RegAddr.B.BHigh=SIDAT;
				TWI0Inx++;
			}
			else
			{
				TWI0RegAddr.B.BLow=SIDAT;
				bTWI0ADFlag=1;					// The next	byte is	Data
				TWI0Inx=0;
			}
		}
		else
		{
			TWI0Buf[TWI0Inx]=SIDAT;				// Save	received data into buffer
			TWI0Inx++;
			if(TWI0Inx>=TEST_BUF_SIZE)
			{
				bTWI0RXFlag=TRUE;
				TWI0Inx=0;
				TWI0OvTime = 0;					// buffer is full,set overtime to zero
			}
		}
	}
	else if((SISTA==0x60)||(SISTA==0x68))
	{//	0x60,0x68: Own SLA+W has been received.	ACK	has	been returned
		bTWI0ADFlag=0;					// The next	two	byte is	Reg	address
		TWI0Inx=0;
		TWI0OvTime = TWI_OV_TIME_MAX;			// restore TWI0	overtime
	}
	else if((SISTA==0xA8)||(SISTA==0xB0))
	{//	0x60,0x68: Own SLA+R has been received.	ACK	has	been returned
		TWI0Inx=0;						// Set buffer index	to zero
		goto _IIC_SET_SIDAT;
	}
	else if((SISTA==0xB8)||(SISTA==0xC0)||(SISTA==0xC8))
	{
		// 0xB8:  Data byte	in SIDAT has been transmitted ACK has been received
		// 0xC0:  Data byte	or Last	data byte in SIDAT has been	transmitted	Not	ACK	has	been received
		// 0xC8:  Last Data	byte in	SIDAT has been transmitted ACK has been	received
_IIC_SET_SIDAT:
		TWI0OvTime = TWI_OV_TIME_MAX;			// restore TWI0	overtime
		if(TWI0Inx>=TEST_BUF_SIZE)
		{
			TWI0Inx=0;
			TWI0OvTime = 0;					// buffer is full,set overtime to zero
		}
		SIDAT=TWI0Buf[TWI0Inx];					// Load	data to	send
		TWI0Inx++;
	}
	IO_TWI0_SCL=0;
	SICON=SICON|(AA);						// Set AA
	SICON=SICON&(~SI);						// clear TWI0 interrupt	flag
	IO_TWI0_SCL=1;

	_pop_(SFRPI);
}

/***********************************************************************************
*Function:		void INT_TWI1(void)
*Description:	TWI1 Interrupt handler

*Input:
*Output:
*************************************************************************************/
void INT_TWI1()	interrupt INT_VECTOR_TWI1
{
	_push_(SFRPI);
	SFR_SetPage(1);
	if(SI1STA==0x80)
	{ // 0x80: DATA	has	been received. ACK has been	returned.
		TWI1OvTime = TWI_OV_TIME_MAX;			// restore TWI1	overtime
		if(bTWI1ADFlag==0)
		{//	Reg	Address
			if(TWI1Inx==0)
			{
				TWI1RegAddr.B.BHigh=SI1DAT;
				TWI1Inx++;
			}
			else
			{
				TWI1RegAddr.B.BLow=SI1DAT;
				bTWI1ADFlag=1;					// The next	byte is	Data
				TWI1Inx=0;
			}
		}
		else
		{
			TWI1Buf[TWI1Inx]=SI1DAT;				// Save	received data into buffer
			TWI1Inx++;
			if(TWI1Inx>=TEST_BUF_SIZE)
			{
				bTWI1RXFlag=TRUE;
				TWI1Inx=0;
				TWI1OvTime = 0;					// buffer is full,set overtime to zero
			}
		}
	}
	else if((SI1STA==0x60)||(SI1STA==0x68))
	{//	0x60,0x68: Own SLA+W has been received.	ACK	has	been returned
		bTWI1ADFlag=0;					// The next	two	byte is	Reg	address
		TWI1Inx=0;
		TWI1OvTime = TWI_OV_TIME_MAX;			// restore TWI1	overtime
	}
	else if((SI1STA==0xA8)||(SI1STA==0xB0))
	{//	0x60,0x68: Own SLA+R has been received.	ACK	has	been returned
		TWI1Inx=0;						// Set buffer index	to zero
		goto _IIC_SET_SI1DAT;
	}
	else if((SI1STA==0xB8)||(SI1STA==0xC0)||(SI1STA==0xC8))
	{
		// 0xB8:  Data byte	in SIDAT has been transmitted ACK has been received
		// 0xC0:  Data byte	or Last	data byte in SIDAT has been	transmitted	Not	ACK	has	been received
		// 0xC8:  Last Data	byte in	SIDAT has been transmitted ACK has been	received
_IIC_SET_SI1DAT:
		TWI1OvTime = TWI_OV_TIME_MAX;			// restore TWI1	overtime
		if(TWI1Inx>=TEST_BUF_SIZE)
		{
			TWI1Inx=0;
			TWI1OvTime = 0;						// buffer is full,set overtime to zero
		}
		SI1DAT=TWI1Buf[TWI1Inx];				// Load	data to	send
		TWI1Inx++;
	}
	IO_TWI1_SCL=0;
	SI1CON=SI1CON|(AA);						// Set AA
	SI1CON=SI1CON&(~SI1);					// clear TWI1 interrupt	flag
	IO_TWI1_SCL=1;
	_pop_(SFRPI);
}

/***********************************************************************************
*Function:		void SendByte(u8 tByte)
*Description:	send byte
*Input:			u8 tByte: the data to be send
*Output:
*************************************************************************************/
void SendByte(u8 tByte)
{
	bES0=ES0;
	ES0=0;
	S0BUF=tByte;
	while(TI0==0);
	TI0=0;
	ES0=bES0;
}

/***********************************************************************************
*Function:		void SendStr(BYTE* PStr)
*Description:	send string
*Input:				u8*	PStr:the string	to be send
*Output:
*************************************************************************************/
void SendStr(u8* PStr)
{
	while(*PStr	!= 0)
	{
		SendByte(*PStr);
		PStr ++;
	}
}

/***********************************************************************************
*Function:		void SendHex(u8	ToSend)
*Description:	send data for HEX
*Input:				u8 ToSend: the data	to be send
*Output:
*************************************************************************************/
void SendHex(u8	ToSend)
{
	u8 x;
	x =	ToSend >> 4;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
	x= ToSend&0x0F;
	if(x < 10)
	{
		SendByte('0'+x);
	}
	else
	{
		SendByte('A'-10+x);
	}
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of	baud rate was S0BRG
*Input:
*Output:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R.	source:	S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: SYSCLK
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_24000000_1T);

	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:		void InitTimer0(void)
*Description:	Initialize Timer0  Sysclk/12
*Input:
*Output:
*************************************************************************************/
void InitTimer0(void)
{
	TM_SetT0Mode_1_16BIT_TIMER();			// Set T0 mode:16bit timer
	TM_SetT0Clock_SYSCLKDiv12();			// Set T0 clock:SYSCLK/12
	TM_SetT0Gate_Disable();

	TM_SetT0LowByte(TIMER_12T_1ms_TL);						// Set T0 low byte
	TM_SetT0HighByte(TIMER_12T_1ms_TH);					// Set T0 high byte

	TM_EnableT0();							// Enable T0
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// Set P30,P31 as Quasi-Bi
	PORT_SetP1OpenDrainPu(BIT0|BIT1);				// Set P10,P11 as open-drain with pull-high(TWI1:SCL/SDA)
	PORT_SetP4OpenDrainPu(BIT0|BIT1);				// Set P40,P41 as open-drain with pull-high(TWI0:SCL/SDA)
}

/***********************************************************************************
*Function:	 void InitTWI0_Slave(void)
*Description:Initialize	TWI0 Slave mode
*Input:
*Output:
*************************************************************************************/
void InitTWI0_Slave()
{
	TWI0_Clear();
	TWI0_SetClock(TWI0_CLK_SYSCLK_DIV_60);				// SYSCLK/60=24MHz/60=400KHz
	TWI0_SetTWI0SCLTWI0SDA_P40P41();
	TWI0_Enable();
	TWI0_SetSlaveAddr(SLAVE_ADDRESS);
	TWI0_SendACK();

}

/***********************************************************************************
*Function:	 void InitTWI1_Slave(void)
*Description:Initialize	TWI1 Slave mode
*Input:
*Output:
*************************************************************************************/
void InitTWI1_Slave()
{
	TWI1_Clear();
	TWI1_SetClock((TWI0_CLK_SYSCLK_DIV_60));				// SYSCLK/60=24MHz/60=400KHz
	TWI1_SetTWI1SCLTWI1SDA_P10P11();
	TWI1_Enable();
	TWI1_SetSlaveAddr(SLAVE_ADDRESS);
	TWI1_SendACK();

}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		void InitInterrupt()
*Description:	Initialize Interrupt
*Input:
*Output:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnTIMER0();				// Enable Timer0 interrupt
	INT_EnTWI0();				// Enable TWI0 interrupt
	INT_EnTWI1();				// Enable TWI1 interrupt
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitTWI0_Slave();
	InitTWI1_Slave();
	InitTimer0();
	InitUart0_S0BRG();
	InitInterrupt();

	INT_EnAll();			//	Enable global interrupt
}

void main()
{
	u8 i;
	InitSystem();


	SendStr("\nStart");

	IO_LED_G=0;IO_LED_Y=0;IO_LED_R=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_Y=1;IO_LED_R=1;

	TWI0OvTime = 0;					// Set buffer index	to zero
	TWI1OvTime = 0;					// Set buffer index	to zero

	while(1)
	{
		if(LedTime==0)
		{
			IO_LED_G = !IO_LED_G;
			if(IO_LED_G==0)
			{
				LedTime=20;
			}
			else
			{
				LedTime=250;
			}
		}
		if(bTWI0RXFlag==TRUE)
		{
			IO_LED_R=0;IO_LED_G=1;
			SendStr("\nTWI0	RX 0x");
			SendHex(TWI0RegAddr.B.BHigh);
			SendHex(TWI0RegAddr.B.BLow);
			for(i=0;i<TEST_BUF_SIZE;i++)
			{
				SendByte(' ');
				SendHex(TWI0Buf[i]);
			}
			bTWI0RXFlag=FALSE;
			IO_LED_R=1;IO_LED_G=1;
			LedTime=250;
		}
		if(bTWI1RXFlag==TRUE)
		{
			IO_LED_Y=0;IO_LED_G=1;
			SendStr("\nTWI1	RX 0x");
			SendHex(TWI1RegAddr.B.BHigh);
			SendHex(TWI1RegAddr.B.BLow);
			for(i=0;i<TEST_BUF_SIZE;i++)
			{
				SendByte(' ');
				SendHex(TWI1Buf[i]);
			}
			bTWI1RXFlag=FALSE;
			IO_LED_Y=1;IO_LED_G=1;
			LedTime=250;
		}

		if(bTWI0Error==TRUE)
		{
			bTWI0Error=FALSE;
			// error,reset TWI0
			IO_LED_R=0;IO_LED_G=0;
			InitTWI0_Slave();
			DelayXms(50);
			IO_LED_R=1;IO_LED_G=1;
			LedTime=250;
		}
		if(bTWI1Error==TRUE)
		{
			bTWI1Error=FALSE;
			// error,reset TWI1
			IO_LED_Y=0;IO_LED_G=0;
			InitTWI1_Slave();
			DelayXms(50);
			IO_LED_Y=1;IO_LED_G=1;
			LedTime=250;
		}
	}
}


